//*************************************************************************************************
//
//	Description:
//		HDRMethods.fx - Various techniques and shaders which are used to perform HDR rendering.
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     01/11/2007  0.1           Created
//		TNettleship     01/22/2007  0.2           Fixed bilinear problems with fp16 targets on NV7x
//																							series.
//		BIrvine			    05/24/2007  0.3           Clamped bottom end of luminance calculation result
//																							to avoid issues with negative light.
//		TNettleship			05/29/2007	0.4						Added support for viewports.
//		TNettleship			05/29/2007	0.5						Added support for gradual exposure adaptation.
//		BIrvine					06/08/2007	0.6						Added temporary code to clamp high luminance values.
//		TMann						05/07/2007	0.7						Added PS3 support and edge detection
//		TNettleship			06/26/2007	0.8						Improved the pixel luminance calc & added clamping
//																							to the range of the tone mapping effect.
//		TNettleship     07/24/2007  0.9           Made sure samplers aren't using anisotropic filtering.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"


//------------------------------------------------------------------
//
//  GLOBAL VARIABLES
//



//------------------------------------------------------------------
//
// Viewport into source framebuffer
//

float2 viewportOrigin;
float2 viewportScale;

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
float luminanceScale;
#else
#define _MINFILTER	Linear
#endif

//------------------------------------------------------------------
//
// Luminance calculation
// 
float2 luminanceGreyscaleOffsets[ 4 ];
float2 luminanceDownsampleOffsets[ 9 ];

texture luminanceInputTexture : TEXTURE;
sampler luminanceInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < luminanceInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture luminanceBilinearInputTexture : TEXTURE;
sampler luminanceBilinearInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < luminanceBilinearInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

float convergenceFactor;
texture luminanceConvergenceTexture : TEXTURE;
sampler luminanceConvergence : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < luminanceConvergenceTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


//------------------------------------------------------------------
//
// Bloom calculation
//
float bloomLowerThreshold;										// Values greater than this are accepted for the bright pass
float bloomUpperThreshold;										// Values greater than this are maximum
float bloomFactor;											// Used in the post-processing, but also useful here
float2 bloomThresholdOffsets[ 4 ];			// Sampling offsets used by the bloom threshold pass
#ifdef _PS3_
float2 bloomDownsampleOffsets[ 4 ];		// Sampling offsets used by the bloom downsample pass
#else
float2 bloomDownsampleOffsets[ 16 ];		// Sampling offsets used by the bloom downsample pass
#endif

float horizBlurWeights[ 9 ];						// Description of the sampling distribution used by
float horizBlurOffsets[ 9 ];						// the HorizontalBlur() function

float vertBlurWeights[ 9 ];							// Description of the sampling distribution used by
float vertBlurOffsets[ 9 ];							// the VerticalBlur() function

texture bloomBilinearGammaInputTexture : TEXTURE;
sampler bloomBilinearGammaInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < bloomBilinearGammaInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	// Uses bilinear filter on ps3
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture sceneGammaInputTexture : TEXTURE;
sampler sceneGammaInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < sceneGammaInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = _MINFILTER;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture bloomBilinearInputTexture : TEXTURE;
sampler bloomBilinearInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < bloomBilinearInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	// Uses bilinear filter on ps3
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture bloomInputTexture : TEXTURE;
sampler bloomInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < bloomInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};



//------------------------------------------------------------------
//
// Vertex shader
//

struct VSINPUT
{
	float3 position : POSITION;								// Texture coords as a vertex position
	float2 texCoord : TEXCOORD0;
};

struct VSOUTPUT
{
	float4 position : POSITION;
	float2 texCoord : TEXCOORD0;
};


VSOUTPUT HDRVertexShader_WithViewport( VSINPUT _input )
{
	VSOUTPUT output;

	// Apply the viewport transformation to the input tex coord
	output.position = float4( _input.position.xyz, 1.0f );
	output.texCoord = ( _input.texCoord * viewportScale ) + viewportOrigin;

	return output;
}



VSOUTPUT HDRVertexShader( VSINPUT _input )
{
	VSOUTPUT output;

	// Just pass through the texture values to the pixel shader
	output.position = float4( _input.position.xyz, 1.0f );
	output.texCoord = _input.texCoord;

	return output;
}



//
// Pixel shaders
//

struct PSINPUT
{
	float2 texCoord : TEXCOORD0;
};

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour : COLOR0;
};


//------------------------------------------------------------------
//
// Function calculates the luminance of a HDR pixel's colour.
//
//------------------------------------------------------------------

float CalculatePixelLuminance( float4 _colour )
{
	const float3 weights = { 0.29f, 0.50f, 0.27f };
	// Clamp colours to PS3 range
//	_colour.r=min(_colour.r, 16.0f);
//	_colour.g=min(_colour.g, 16.0f);
//	_colour.b=min(_colour.b, 16.0f);
	float lLuminance = dot( _colour.rgb, weights );

	return lLuminance;
}

//------------------------------------------------------------------
//
// First pass of the luminance calculation process.
//
// Downsamples the HDR source texture 2x2, and calculates
// the average and maximum luminance values.
//
// Writes the average and max luminance to the 2 channel output
// buffer.
//
//------------------------------------------------------------------

PSOUTPUT Luminance_Greyscale_FragmentShader( PSINPUT _input )
{
#ifdef _PS3_
	// PS3 uses bilinear
	const float4 colour = tex2D( luminanceBilinearInput, _input.texCoord );
	float average = colour.a;
	float _log=average;
#else

	float4 colour;
	float4 luminances;

	// Stuff 4 lum samples into the vector
	colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 0 ].x, luminanceGreyscaleOffsets[ 0 ].y ) );
	luminances.x = CalculatePixelLuminance( colour );
	colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 1 ].x, luminanceGreyscaleOffsets[ 1 ].y ) );
	luminances.y = CalculatePixelLuminance( colour );
	colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 2 ].x, luminanceGreyscaleOffsets[ 2 ].y ) );
	luminances.z = CalculatePixelLuminance( colour );
	colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 3 ].x, luminanceGreyscaleOffsets[ 3 ].y ) );
	luminances.w = CalculatePixelLuminance( colour );

	float average = ( luminances.x+luminances.y+luminances.z+luminances.w ) * 0.25f;
	
	// Calc max
	float scale=(2.7f-1.0f)/16.0f;
	float _log = log( average*scale+1.0f+0.001f );

	_log=clamp(_log, 0.0f, 1.0f);
	average=clamp(average, 0.0f, 20.0f);
	
#endif

	// Output the luminance to the render target
	PSOUTPUT output;
	output.colour = float4( _log, average, average, 0.0f );
	return output;
}



//------------------------------------------------------------------
//
// Subsequent passes of the luminance calculation perform a 3x3
// downsampling of a two channel avg/max luminance input to produce
// a similar output.
//
//------------------------------------------------------------------

#ifdef _PS3_
PSOUTPUT Luminance_Bilinear_Downsample_FragmentShader( PSINPUT _input )
{
	// Again PS3 uses bilinear
	float average = 0.0f;

	// Sample the source texture 4 times, using precalculated offsets
	for( int i = 0; i < 4; i++ )
	{
		const float4 colour = tex2D( luminanceBilinearInput, _input.texCoord + float2( luminanceDownsampleOffsets[ i ].x, luminanceDownsampleOffsets[ i ].y ) );
		average += colour.r;
	}

	// Calculate the average of the samples
	average *=0.25f;

	// Output the average and max luminance to the render target
	PSOUTPUT output;
	output.colour = float4( average, average, average, 1.0f );
	return output;
}
#endif

PSOUTPUT Luminance_Downsample_FragmentShader( PSINPUT _input )
{
	float average = 0.0f;
	float count = 0.0f;

	// Sample the source texture 9 times, using precalculated offsets
	for( int i = 0; i < 9; i++ )
	{
		const float4 colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceDownsampleOffsets[ i ].x, luminanceDownsampleOffsets[ i ].y ) );
		float luminance = max( 0.0f, colour.r );
		average += luminance;
		count += sign( luminance );
	}

	// Calculate the average of the samples
	average /= max( count, 1.0f );

	// Output the average and max luminance to the render target
	PSOUTPUT output;
	output.colour = float4( average, average, average, 1.0f );
	return output;
}



//------------------------------------------------------------------
//
// Subsequent passes of the luminance calculation perform a 3x3
// downsampling of a two channel avg/max luminance input to produce
// a similar output.
//
//------------------------------------------------------------------

PSOUTPUT Luminance_Convergence_FragmentShader( PSINPUT _input )
{
	PSOUTPUT output;

	// Read this frame's calculated luminance values and the "current" values
	float thisFrame = tex2D( luminanceInput, float2( 0.5f, 0.5f ) ).r;
	float currentValues = tex2D( luminanceConvergence, float2( 0.5f, 0.5f ) ).r;

	float lerpResult = ( ( currentValues * ( 1.0f - convergenceFactor ) ) + ( thisFrame * convergenceFactor ) );

	// Converge the current value towards the calculated values
	output.colour = float4( lerpResult, lerpResult, lerpResult, lerpResult );
	return output;
}


//------------------------------------------------------------------
//
// First pass of the masked luminance calculation process.
//
// Based on Luminance_Greyscale_FragmentShader(), adds detection for 2x2 quads with all samples masked.
// Also passed weight of the valid samples to next level.
//
//------------------------------------------------------------------
PSOUTPUT MaskedLuminance_Greyscale_FragmentShader( PSINPUT _input )
{
#ifdef _PS3_

	// PS3 uses bilinear
	const float4 colour = tex2D( luminanceBilinearInput, _input.texCoord );
	const float averageLogLuminance = colour.a;
	// If all four samples were mask (0), make the weight 0, otherwise 1.
	const float quadWeight = sign( colour.a );

#else	// ! _PS3_

	float4 luminances;

	// Stuff 4 lum samples into the vector
	const float4 colour1 = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 0 ].x, luminanceGreyscaleOffsets[ 0 ].y ) );
	luminances.x = CalculatePixelLuminance( colour1 );
	const float4 colour2 = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 1 ].x, luminanceGreyscaleOffsets[ 1 ].y ) );
	luminances.y = CalculatePixelLuminance( colour2 );
	const float4 colour3 = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 2 ].x, luminanceGreyscaleOffsets[ 2 ].y ) );
	luminances.z = CalculatePixelLuminance( colour3 );
	const float4 colour4 = tex2D( luminanceInput, _input.texCoord + float2( luminanceGreyscaleOffsets[ 3 ].x, luminanceGreyscaleOffsets[ 3 ].y ) );
	luminances.w = CalculatePixelLuminance( colour4 );

	const float average = ( luminances.x + luminances.y + luminances.z + luminances.w ) / 4.0f;

	const float scale = ( 2.7f - 1.0f ) / 16.0f;
	float _log = log( average * scale + 1.0f + 0.001f );

	const float averageLogLuminance = clamp( _log, 0.0f, 1.0f );

	// Compute how many non-zero (unmasked) samples there were.
	const float quadWeight = ( sign( colour1.a ) + sign( colour2.a ) + sign( colour3.a ) + sign( colour4.a ) ) / 4.0f;

#endif

	PSOUTPUT output;
	output.colour = COLOUR_OUTPUT_TYPE( averageLogLuminance, quadWeight, 0.0f, 1.0f );
	return output;
}


//------------------------------------------------------------------
//
// Second and subsequent passes of the masked luminance calculation process.
//
// Based on Luminance_Downsample_FragmentShader(), adds processing of weighted average.
//
//------------------------------------------------------------------
PSOUTPUT MaskedLuminance_Downsample_FragmentShader( PSINPUT _input )
{
	float sum = 0.0f;
	float weight = 0.0f;

	// Sample the source texture 9 times, using precalculated offsets
	for( int i = 0; i < 9; i++ )
	{
		const float4 colour = tex2D( luminanceInput, _input.texCoord + float2( luminanceDownsampleOffsets[ i ].x, luminanceDownsampleOffsets[ i ].y ) );
		const float sampleLuminance = max( 0.0f, colour.r );
		const float sampleWeight = colour.g;
		sum += sampleLuminance * sampleWeight;
		weight += sampleWeight;
	}

	// Calculate the weighted average, and normalize the summed sample weight.

	const float average = ( ( weight > 0.001f ) ? ( sum / weight ) : sum );
	weight /= 9.0f;

	PSOUTPUT output;
	output.colour = COLOUR_OUTPUT_TYPE( average, weight, 0.0f, 1.0f );
	return output;
}


//------------------------------------------------------------------
//
// First pass of the bloom calculation process.
//
// Downsamples the source HDR colour texture 2x2 and performs a
// threshold calculation to decide if the average pixel colour
// is bright enough. If so, writes it to the output buffer. If not,
// writes black.
//
//------------------------------------------------------------------
PSOUTPUT Bloom_Threshold_FragmentShader( PSINPUT _input )
{
#ifdef _PS3_
	float4 colour;

	// Downsample the source texture 2x2
	colour=tex2D( bloomBilinearGammaInput, _input.texCoord );

	// Calculate the luminance of the average colour

	float scale=(2.7f-1.0f)/luminanceScale;
    float lum = (exp(colour.a)-1.0f)/scale;

	colour *= smoothstep(bloomLowerThreshold, bloomUpperThreshold, lum) * bloomFactor;

	// Output the colour to the render target
	PSOUTPUT output;
	output.colour = colour;
	return output;
#else
	float4 average = { 0.0f, 0.0f, 0.0f, 1.0f };
	float4 colour;

	// Downsample the source texture 2x2
	for( int i = 0; i < 4; i++ )
	{
		colour=tex2D( bloomInput, _input.texCoord + float2( bloomThresholdOffsets[ i ].x, bloomThresholdOffsets[ i ].y ) );
		average += colour;
	}
	average *= 0.25f;

	// Calculate the luminance of the average colour
	float luminance = CalculatePixelLuminance( average );

	// Need to encode as log to try and emulate PS3
	
	// Convert to u8
	float scale=(2.7f-1.0f)/16.0f;
	float _log = log( luminance*scale+1.0f+0.001f );
	_log=clamp(_log, 0.0f, 1.0f);
	_log*=255.0f;
	int Ilog=(int)_log;
	
	// Now convert back to float
	_log=(float)Ilog;
	_log/=255.0f;
	luminance=(exp(_log)-1.0f)/scale;
	
	average *= (smoothstep( bloomLowerThreshold, bloomUpperThreshold, luminance ) * bloomFactor) * 0.75f;
	
	// Output the colour to the render target
	PSOUTPUT output;
	output.colour = average;
	return output;
#endif	
}
    
    

//------------------------------------------------------------------
//
// Second pass of the bloom calculation process.
//
// Downsamples the result of the first pass a further 4x4 times.
//
//------------------------------------------------------------------

PSOUTPUT Bloom_Downsample_FragmentShader( PSINPUT _input )
{
#ifdef _PS3_
	float4 average = { 0.0f, 0.0f, 0.0f, 0.0f };

	// Calculate the average of 4 (bilinear) input pixels
	for( int i = 0; i < 4; i++ )
	{
		average += tex2D( bloomBilinearInput, _input.texCoord + float2( bloomDownsampleOffsets[ i ].x, bloomDownsampleOffsets[ i ].y ) );
	}

	average *= ( 1.0f / 4.0f );
#else
	float4 average = { 0.0f, 0.0f, 0.0f, 0.0f };

	// Calculate the average of 16 input pixels
	for( int i = 0; i < 16; i++ )
	{
		average += tex2D( bloomInput, _input.texCoord + float2( bloomDownsampleOffsets[ i ].x, bloomDownsampleOffsets[ i ].y ) );
	}

	average *= ( 1.0f / 16.0f );
#endif	

	// Output the colour to the render target
	PSOUTPUT output;
	output.colour = average;
	return output;
}
    
    

//------------------------------------------------------------------
// HORIZONTAL BLUR
//
// Takes 9 samples from the down-sampled texture (4 either side and
// one central) biased by the provided weights. Different weight
// distributions will give more subtle/pronounced blurring.
//------------------------------------------------------------------
PSOUTPUT Bloom_HorizBlur_FragmentShader( PSINPUT _input )
{
	float4 colour = { 0.0f, 0.0f, 0.0f, 0.0f };

	for( int i = 0; i < 9; i++ )
	{
		colour += ( tex2D( bloomInput, _input.texCoord + float2( horizBlurOffsets[ i ], 0.0f ) ) * horizBlurWeights[ i ] );
	}

	// Output the colour to the render target
	PSOUTPUT output;
	output.colour = float4( colour.rgb, 1.0f );
	return output;
}
    
    
    
//------------------------------------------------------------------
// VERTICAL BLUR
//
// Takes 9 samples from the down-sampled texture (4 above/below and
// one central) biased by the provided weights. Different weight
// distributions will give more subtle/pronounced blurring.
//------------------------------------------------------------------
PSOUTPUT Bloom_VertBlur_FragmentShader( PSINPUT _input )
{
	float4 colour = { 0.0f, 0.0f, 0.0f, 0.0f };

	for( int i = 0; i < 9; i++ )
	{
		colour += ( tex2D( bloomInput, _input.texCoord + float2( 0.0f, vertBlurOffsets[ i ] ) ) * vertBlurWeights[ i ] );
	}

	// Output the colour to the render target
	PSOUTPUT output;
	output.colour = float4( colour.rgb, 1.0f );
	return output;
}

#ifdef _PS3_
//------------------------------------------------------------------
//  SHADER ENTRY POINT
//------------------------------------------------------------------
PSOUTPUT HDR_AddBloom_FragmentShader( PSINPUT _input )
{
	// Apply the the viewport xform to the input texcoord, to get the bloom texcoord
	float2 bloomCoord = ( _input.texCoord - viewportOrigin ) / viewportScale;

	// Sample input texture
	float4 scene=tex2D ( sceneGammaInput, bloomCoord );

	// Sample bloom texture	
	float4 final=tex2D( bloomBilinearInput, bloomCoord );

	// Return the fully composed colour
	PSOUTPUT output;
	output.colour = float4( scene.rgb+final.rgb, 1.0f );
	return output;
}
#endif

#ifdef _PS3_
technique HDR_AddBloom
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx HDR_AddBloom_FragmentShader();
	}
}
#endif

technique Luminance_Greyscale
<
	int regcount=38;
>
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader_WithViewport();
		PixelShader = compile sce_fp_rsx Luminance_Greyscale_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader_WithViewport();
		PixelShader = compile ps_3_0 Luminance_Greyscale_FragmentShader();
#endif		
	}
}


technique Luminance_Downsample
<
	int regcount=48;
>
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx Luminance_Downsample_FragmentShader();
#else
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 Luminance_Downsample_FragmentShader();
#endif		
	}
}


technique Luminance_Convergence
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_		
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx Luminance_Convergence_FragmentShader();
#else
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 Luminance_Convergence_FragmentShader();
#endif		
	}
}


technique MaskedLuminance_Greyscale
<
	int regcount=38;
>
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader_WithViewport();
		PixelShader = compile sce_fp_rsx MaskedLuminance_Greyscale_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader_WithViewport();
		PixelShader = compile ps_3_0 MaskedLuminance_Greyscale_FragmentShader();
#endif		
	}
}


technique MaskedLuminance_Downsample
<
	int regcount=48;
>
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx MaskedLuminance_Downsample_FragmentShader();
#else
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 MaskedLuminance_Downsample_FragmentShader();
#endif		
	}
}


technique Bloom_Threshold
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader_WithViewport();
		PixelShader = compile sce_fp_rsx Bloom_Threshold_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader_WithViewport();
		PixelShader = compile ps_3_0 Bloom_Threshold_FragmentShader();
#endif		
	}
}


technique Bloom_Downsample
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx Bloom_Downsample_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 Bloom_Downsample_FragmentShader();
#endif		
	}
}


technique Bloom_HorizBlur
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx Bloom_HorizBlur_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 Bloom_HorizBlur_FragmentShader();
#endif		
	}
}


technique Bloom_VerticalBlur
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx Bloom_VertBlur_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 Bloom_VertBlur_FragmentShader();
#endif		
	}
}

